/*
 * Decompiled with CFR 0.152.
 */
package com.kpouer.mapview;

import com.kpouer.mapview.LatLng;
import com.kpouer.mapview.MapPoint;
import com.kpouer.mapview.marker.Marker;
import com.kpouer.mapview.tile.TileServer;
import com.kpouer.mapview.tile.TilesTools;
import com.kpouer.mapview.widget.MouseLocationLabel;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import javax.swing.JPanel;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MapView
extends JPanel {
    private static final Logger log = LoggerFactory.getLogger(MapView.class);
    private TileServer tileServer;
    private final MouseLocationLabel mouseLocationLabel;
    private int xPos;
    private int yPos;
    private int zoom = 13;
    private final TilesTools tilesTools;
    private final List<Marker> markers;

    public MapView(TileServer tileServer) {
        super(null);
        this.tileServer = tileServer;
        this.tilesTools = new TilesTools(tileServer.getTilesSize());
        this.markers = new ArrayList<Marker>();
        this.mouseLocationLabel = new MouseLocationLabel();
        this.setCenter(48.85337, 2.34847, this.zoom);
        this.addComponentListener(new ComponentAdapter(){

            @Override
            public void componentResized(ComponentEvent e) {
                MapView.this.markers.forEach(marker -> marker.computeLocation(MapView.this));
                MapView.this.mouseLocationLabel.setLocation(MapView.this.getWidth() - MapView.this.mouseLocationLabel.getWidth(), MapView.this.getHeight() - MapView.this.mouseLocationLabel.getHeight());
            }
        });
        MyMouseAdapter mouseAdapter = new MyMouseAdapter();
        this.addMouseWheelListener(mouseAdapter);
        this.addMouseListener(mouseAdapter);
        this.addMouseMotionListener(mouseAdapter);
        this.add(this.mouseLocationLabel);
    }

    public void setTileServer(TileServer tileServer) {
        if (this.tileServer != tileServer) {
            log.info("setTileServer: {}", (Object)tileServer);
            this.tileServer = tileServer;
        }
    }

    public void setMouseLocationLabelVisible(boolean visible) {
        this.mouseLocationLabel.setVisible(visible);
    }

    public void addMarker(Marker marker) {
        this.markers.add(marker);
        marker.computeLocation(this);
    }

    public void removeMarker(Marker marker) {
        this.markers.remove(marker);
    }

    public void removeAllMarkers() {
        this.markers.clear();
    }

    public boolean areMarkersVisible(List<? extends Marker> markers) {
        double minLatitude = this.pointScreenToLatitude(this.getHeight());
        double minLongitude = this.pointScreenToLongitude(0);
        double maxLatitude = this.pointScreenToLatitude(0);
        double maxLongitude = this.pointScreenToLongitude(this.getWidth());
        return markers.stream().filter(marker -> !this.isMarkerWithinBounds((Marker)marker, minLatitude, minLongitude, maxLatitude, maxLongitude)).findFirst().isEmpty();
    }

    public boolean isMarkerVisible(Marker marker) {
        double minLatitude = this.pointScreenToLatitude(0);
        double minLongitude = this.pointScreenToLongitude(0);
        double maxLatitude = this.pointScreenToLatitude(this.getHeight());
        double maxLongitude = this.pointScreenToLongitude(this.getWidth());
        return this.isMarkerWithinBounds(marker, minLatitude, minLongitude, maxLatitude, maxLongitude);
    }

    public boolean isMarkerWithinBounds(Marker marker, double minLatitude, double minLongitude, double maxLatitude, double maxLongitude) {
        return marker.getLatitude() > minLatitude && marker.getLatitude() < maxLatitude && marker.getLongitude() > minLongitude && marker.getLongitude() < maxLongitude;
    }

    public void fitToMarkersIfNecessary() {
        this.fitToMarkersIfNecessary(this.markers);
    }

    public void fitToMarkersIfNecessary(List<? extends Marker> markers) {
        if (!this.areMarkersVisible(markers)) {
            this.fitToMarkers(markers);
        }
    }

    public void fitToMarkers() {
        this.fitToMarkers(this.markers);
    }

    private void moveMarker(Marker marker, int x, int y) {
        marker.setLongitude(this.tilesTools.position2longitude(x + this.getShiftX(), this.zoom));
        marker.setLatitude(this.tilesTools.position2latitude(y + this.getShiftY(), this.zoom));
        marker.computeLocation(this);
    }

    public void fitToMarkers(List<? extends Marker> markers) {
        int zoom;
        if (markers.isEmpty()) {
            return;
        }
        Marker firstMarker = markers.get(0);
        double minLat = firstMarker.getLatitude();
        double minLon = firstMarker.getLongitude();
        double maxLat = firstMarker.getLatitude();
        double maxLon = firstMarker.getLongitude();
        for (int i = 1; i < markers.size(); ++i) {
            Marker marker = markers.get(i);
            minLat = Math.min(minLat, marker.getLatitude());
            minLon = Math.min(minLon, marker.getLongitude());
            maxLat = Math.max(maxLat, marker.getLatitude());
            maxLon = Math.max(maxLon, marker.getLongitude());
        }
        int width = this.getWidth();
        int height = this.getHeight();
        for (zoom = this.tileServer.getMinZoom(); zoom < this.tileServer.getMaxZoom(); ++zoom) {
            int minX = this.tilesTools.longitudeToPoint(minLon, zoom);
            int minY = this.tilesTools.latitudeToPoint(minLat, zoom);
            int maxX = this.tilesTools.longitudeToPoint(maxLon, zoom);
            int maxY = this.tilesTools.latitudeToPoint(maxLat, zoom);
            int diffX = Math.abs(maxX - minX);
            int diffY = Math.abs(maxY - minY);
            if (2 * diffX > width || 2 * diffY > height) break;
        }
        this.setCenter(minLat + (maxLat - minLat) / 2.0, minLon + (maxLon - minLon) / 2.0, zoom);
    }

    public void setZoom(int zoom) {
        this.xPos = this.tilesTools.zoom(this.xPos, this.zoom, zoom);
        this.yPos = this.tilesTools.zoom(this.yPos, this.zoom, zoom);
        this.zoom = this.clipZoom(zoom);
        this.markers.forEach(marker -> marker.computeLocation(this));
    }

    public void setCenter(int xPos, int yPos) {
        this.xPos = xPos;
        this.yPos = yPos;
        this.markers.forEach(marker -> marker.computeLocation(this));
    }

    public void setCenter(int xPos, int yPos, int zoom) {
        this.xPos = xPos;
        this.yPos = yPos;
        this.zoom = zoom;
        this.markers.forEach(marker -> marker.computeLocation(this));
    }

    public double getLatitude() {
        return this.tilesTools.position2latitude(this.yPos, this.zoom);
    }

    public double getLongitude() {
        return this.tilesTools.position2longitude(this.xPos, this.zoom);
    }

    public LatLng getCenter() {
        return new LatLng(this.getLatitude(), this.getLongitude());
    }

    public void setCenter(LatLng latLng) {
        this.setCenter(latLng.getLat(), latLng.getLon());
    }

    public void setCenter(LatLng latLng, int zoom) {
        this.setCenter(latLng.getLat(), latLng.getLon(), zoom);
    }

    public void setCenter(double latitude, double longitude) {
        this.setCenter(this.tilesTools.longitudeToPoint(longitude, this.zoom), this.tilesTools.latitudeToPoint(latitude, this.zoom));
    }

    public void setCenter(double latitude, double longitude, int zoom) {
        this.setCenter(this.tilesTools.longitudeToPoint(longitude, zoom), this.tilesTools.latitudeToPoint(latitude, zoom), zoom);
    }

    public MapPoint getCenterPoint() {
        return new MapPoint(this.xPos, this.yPos, this.zoom);
    }

    private LatLng mapPointToLatLng(MapPoint mapPoint) {
        return new LatLng(this.tilesTools.position2latitude(mapPoint.getY(), mapPoint.getZoom()), this.tilesTools.position2longitude(mapPoint.getX(), mapPoint.getZoom()));
    }

    private MapPoint latLngToMapPoint(LatLng latLng, int newZoom) {
        return new MapPoint(this.tilesTools.longitudeToPoint(latLng.getLon(), newZoom), this.tilesTools.latitudeToPoint(latLng.getLat(), newZoom), newZoom);
    }

    public int latitudeToPointScreen(double latitude) {
        return this.tilesTools.latitudeToPoint(latitude, this.zoom) - this.getShiftY();
    }

    public int longitudeToPointScreen(double longitude) {
        return this.tilesTools.longitudeToPoint(longitude, this.zoom) - this.getShiftX();
    }

    private double pointScreenToLatitude(int y) {
        return this.tilesTools.position2latitude(y + this.getShiftY(), this.zoom);
    }

    private double pointScreenToLongitude(int x) {
        return this.tilesTools.position2longitude(x + this.getShiftX(), this.zoom);
    }

    public LatLng pointScreenToLatLng(Point point) {
        return this.pointScreenToLatLng(point, new LatLng());
    }

    public LatLng pointScreenToLatLng(Point point, LatLng target) {
        target.setLat(this.pointScreenToLatitude(point.y));
        target.setLon(this.pointScreenToLongitude(point.x));
        return target;
    }

    public MapPoint pointScreenToMapPoint(Point point) {
        return new MapPoint(point.x + this.getShiftX(), point.y + this.getShiftY(), this.zoom);
    }

    public void translate(int x, int y) {
        this.setCenter(this.xPos + x, this.yPos + y);
    }

    private int getShiftX() {
        return this.xPos - this.getWidth() / 2;
    }

    private int getShiftY() {
        return this.yPos - this.getHeight() / 2;
    }

    @Override
    public void paintComponent(Graphics g) {
        super.paintComponent(g);
        int realCenterX = this.getWidth() / 2;
        int realCenterY = this.getHeight() / 2;
        int shiftX = this.xPos - realCenterX;
        int shiftY = this.yPos - realCenterY;
        int tilesSize = this.tileServer.getTilesSize();
        int centerTileX = realCenterX - this.xPos % tilesSize;
        int centerTileY = realCenterY - this.yPos % tilesSize;
        int leftXTiles = centerTileX / tilesSize + 1;
        int topYTiles = centerTileY / tilesSize + 1;
        int startX = centerTileX - leftXTiles * tilesSize;
        int startY = centerTileY - topYTiles * tilesSize;
        boolean hasToRepaint = false;
        for (int x = startX; x < this.getWidth(); x += tilesSize) {
            for (int y = startY; y < this.getHeight(); y += tilesSize) {
                try {
                    Image image = this.tileServer.getTile((x + shiftX) / tilesSize, (y + shiftY) / tilesSize, this.zoom);
                    if (image != null) {
                        g.drawImage(image, x, y, null);
                        continue;
                    }
                    hasToRepaint = true;
                    continue;
                }
                catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        if (hasToRepaint) {
            this.repaint();
        }
        this.markers.forEach(marker -> marker.paint(g));
    }

    @Override
    protected void processMouseEvent(MouseEvent e) {
        super.processMouseEvent(e);
        Point point = e.getPoint();
        this.markers.stream().filter(marker -> marker.contains(point)).forEach(marker -> marker.processMouseEvent(e));
    }

    @Override
    protected void processMouseMotionEvent(MouseEvent e) {
        super.processMouseMotionEvent(e);
        Point point = e.getPoint();
        this.markers.stream().filter(marker -> marker.contains(point)).forEach(marker -> marker.processMouseMotionEvent(e));
    }

    private int clipZoom(int zoom) {
        return this.tilesTools.clip(zoom, this.tileServer.getMinZoom(), this.tileServer.getMaxZoom());
    }

    public int getZoom() {
        return this.zoom;
    }

    private class MyMouseAdapter
    extends MouseAdapter {
        @Nullable
        private Point startDrag;
        @Nullable
        private Marker draggingMarker;

        private MyMouseAdapter() {
        }

        @Override
        public void mouseMoved(MouseEvent e) {
            MapView.this.mouseLocationLabel.setLatLng(MapView.this.pointScreenToLatLng(e.getPoint(), MapView.this.mouseLocationLabel.getLatLng()));
            MapView.this.repaint();
        }

        @Override
        public void mousePressed(MouseEvent e) {
            Point point = e.getPoint();
            Optional<Marker> first = MapView.this.markers.stream().filter(Marker::isDraggable).filter(marker -> marker.contains(point)).findFirst();
            first.ifPresent(marker -> {
                this.draggingMarker = marker;
            });
        }

        @Override
        public void mouseReleased(MouseEvent e) {
            this.startDrag = null;
            this.draggingMarker = null;
        }

        @Override
        public void mouseDragged(MouseEvent e) {
            if (this.startDrag == null) {
                this.startDrag = e.getPoint();
            } else {
                if (this.draggingMarker == null) {
                    MapView.this.translate(this.startDrag.x - e.getX(), this.startDrag.y - e.getY());
                } else {
                    MapView.this.moveMarker(this.draggingMarker, e.getX(), e.getY());
                }
                this.startDrag = e.getPoint();
                MapView.this.repaint();
            }
        }

        @Override
        public void mouseWheelMoved(MouseWheelEvent e) {
            int newZoom = MapView.this.clipZoom(MapView.this.zoom - e.getWheelRotation());
            if (MapView.this.zoom == newZoom) {
                return;
            }
            MapView.this.tileServer.cancelPendingZoom(newZoom);
            MapPoint mousePoint = MapView.this.pointScreenToMapPoint(e.getPoint());
            MapPoint newMousePoint = MapView.this.tilesTools.zoom(mousePoint, MapView.this.zoom, newZoom);
            MapView.this.setCenter(newMousePoint.getX() - e.getX() + MapView.this.getWidth() / 2, newMousePoint.getY() - e.getY() + MapView.this.getHeight() / 2, newZoom);
            MapView.this.repaint();
        }
    }
}

