/*
 * Decompiled with CFR 0.152.
 */
package com.kpouer.roadwork.service;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.kpouer.mapview.LatLng;
import com.kpouer.mapview.tile.TileServer;
import com.kpouer.roadwork.configuration.Config;
import com.kpouer.roadwork.model.Roadwork;
import com.kpouer.roadwork.model.RoadworkData;
import com.kpouer.roadwork.model.sync.Status;
import com.kpouer.roadwork.opendata.OpendataService;
import com.kpouer.roadwork.opendata.json.DefaultJsonService;
import com.kpouer.roadwork.opendata.json.model.ServiceDescriptor;
import com.kpouer.roadwork.service.HttpService;
import com.kpouer.roadwork.service.ResourceService;
import com.kpouer.roadwork.service.RoadworkMigrationService;
import com.kpouer.roadwork.service.SynchronizationService;
import com.kpouer.roadwork.service.exception.OpenDataException;
import com.kpouer.roadwork.service.serdes.ShapeSerializer;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;
import javax.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.context.ApplicationContext;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;

@Service
public class OpendataServiceManager {
    private static final Logger logger = LoggerFactory.getLogger(OpendataServiceManager.class);
    private static final String VERSION = "2";
    private final HttpService httpService;
    private final Config config;
    private final ApplicationContext applicationContext;
    private final SynchronizationService synchronizationService;
    private final ResourceService resourceService;
    private final ObjectMapper objectMapper = new ObjectMapper();
    private final Map<String, OpendataService> opendataServices = new HashMap<String, OpendataService>();

    @PostConstruct
    public void postConstruct() {
        this.objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        SimpleModule module = new SimpleModule();
        this.applicationContext.getBeansOfType(ShapeSerializer.class).values().forEach(module::addSerializer);
        this.objectMapper.registerModule(module);
    }

    public List<String> getServices() {
        logger.info("getServices");
        List<String> services = this.getDefaultServices();
        List<String> thirdPartyServices = this.getThirdPartyServices();
        ArrayList<String> allServices = new ArrayList<String>(services.size() + thirdPartyServices.size());
        allServices.addAll(services);
        allServices.addAll(thirdPartyServices);
        return allServices;
    }

    public List<String> getDefaultServices() {
        logger.info("getDefaultServices");
        List<String> services = this.getServicesFromPath("opendata/json/");
        services.forEach(service -> logger.info("Default service : " + service));
        return services;
    }

    public List<String> getThirdPartyServices() {
        logger.info("getThirdPartyServices");
        List<String> services = this.getServicesFromPath("thirdparty");
        services.forEach(service -> logger.info("Third party service : " + service));
        return services;
    }

    private List<String> getServicesFromPath(String folder) {
        block10: {
            List<String> list;
            block9: {
                Path pathFolder = Path.of(folder, new String[0]);
                logger.info("getServicesFromPath {}", (Object)pathFolder.toAbsolutePath());
                Stream<Path> files = Files.list(pathFolder);
                try {
                    List<String> services = files.filter(path -> path.toString().endsWith(".json")).map(Path::toFile).map(File::getName).toList();
                    services.forEach(service -> logger.info("Disk service {}/{}", (Object)folder, service));
                    list = services;
                    if (files == null) break block9;
                }
                catch (Throwable throwable) {
                    try {
                        if (files != null) {
                            try {
                                files.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (NoSuchFileException e) {
                        logger.info("Path {} do not exist", (Object)pathFolder.toAbsolutePath());
                        break block10;
                    }
                    catch (IOException e) {
                        logger.error("Unable to read opendata services", e);
                    }
                }
                files.close();
            }
            return list;
        }
        return Collections.emptyList();
    }

    public Optional<RoadworkData> getData() throws RestClientException, IOException, OpenDataException {
        Optional<RoadworkData> roadworks = this.getRoadworks();
        roadworks.ifPresent(roadworkData -> {
            OpendataServiceManager.applyFinishedStatus(roadworkData);
            this.synchronizationService.synchronize((RoadworkData)roadworkData);
        });
        return roadworks;
    }

    public void save(RoadworkData roadworkData) {
        logger.info("save {}", (Object)roadworkData.getSource());
        Path savePath = this.getPath(roadworkData.getSource());
        try {
            Files.createDirectories(savePath.getParent(), new FileAttribute[0]);
            Files.write(savePath, this.objectMapper.writeValueAsBytes(roadworkData), new OpenOption[0]);
        }
        catch (IOException e) {
            logger.error("Unable to save cache to {}", (Object)savePath, (Object)e);
        }
    }

    public LatLng getCenter() {
        try {
            return this.getOpendataService().getMetadata().getCenter();
        }
        catch (OpenDataException e) {
            return new LatLng(48.85337, 2.34847);
        }
    }

    @NonNull
    private Optional<RoadworkData> getRoadworks() throws RestClientException, IOException, OpenDataException {
        Path currentPath = this.getPath(this.config.getOpendataService());
        logger.info("getData {}", (Object)currentPath);
        Optional<RoadworkData> cachedDataOptional = this.loadCache(currentPath);
        if (cachedDataOptional.isEmpty()) {
            logger.info("There is no cached data");
            Optional<RoadworkData> newData = this.getOpendataService().getData();
            newData.ifPresent(this::save);
            return newData;
        }
        RoadworkData cachedRoadworkData = cachedDataOptional.get();
        if (cachedRoadworkData.getCreated() + 86400000L < System.currentTimeMillis()) {
            logger.info("Cache is obsolete {}", (Object)currentPath);
            Files.delete(currentPath);
            Optional<RoadworkData> newDataOptional = this.getOpendataService().getData();
            if (newDataOptional.isPresent()) {
                RoadworkData newData = newDataOptional.get();
                Map<String, Roadwork> newRoadworks = newData.getRoadworks();
                logger.info("reloaded {} new roadworks", (Object)newRoadworks.size());
                for (Roadwork existingRoadwork : cachedRoadworkData) {
                    Roadwork newRoadwork = newRoadworks.get(existingRoadwork.getId());
                    if (newRoadwork == null) continue;
                    logger.info("Roadwork {} -> status {}", (Object)existingRoadwork.getId(), (Object)existingRoadwork.getSyncData().getStatus());
                    newRoadwork.getSyncData().copy(existingRoadwork.getSyncData());
                    existingRoadwork.updateMarker();
                }
                this.save(newData);
            }
            return newDataOptional;
        }
        return cachedDataOptional;
    }

    @NonNull
    public OpendataService getOpendataService() throws OpenDataException {
        logger.debug("getOpendataService");
        String opendataService = this.config.getOpendataService();
        return this.getOpendataService(opendataService);
    }

    public TileServer getTileServer() {
        String tileServerName;
        try {
            tileServerName = this.getOpendataService().getMetadata().getTileServer();
        }
        catch (OpenDataException e) {
            return this.config.getWazeINTLTileServer();
        }
        try {
            return this.applicationContext.getBean(tileServerName + "TileServer", TileServer.class);
        }
        catch (BeansException e) {
            logger.error("Error getting tile server " + tileServerName, e);
            return this.config.getWazeINTLTileServer();
        }
    }

    @NonNull
    public OpendataService getOpendataService(String opendataService) throws OpenDataException {
        logger.debug("getOpendataService {}", (Object)opendataService);
        OpendataService opendataServiceInstance = this.opendataServices.get(opendataService);
        if (opendataServiceInstance == null) {
            if (opendataService.endsWith(".json")) {
                opendataServiceInstance = this.getJsonService(opendataService);
            } else {
                try {
                    opendataServiceInstance = this.applicationContext.getBean(opendataService, OpendataService.class);
                }
                catch (NoSuchBeanDefinitionException e) {
                    throw new OpenDataException(e.getMessage(), e);
                }
            }
            this.opendataServices.put(opendataService, opendataServiceInstance);
        }
        return opendataServiceInstance;
    }

    @NonNull
    private DefaultJsonService getJsonService(String opendataService) throws OpenDataException {
        logger.info("getJsonService {}", (Object)opendataService);
        try {
            Optional<URL> opendataServicePath = this.resourceService.getResource(opendataService);
            if (opendataServicePath.isPresent()) {
                ServiceDescriptor serviceDescriptor = this.objectMapper.readValue(opendataServicePath.get(), ServiceDescriptor.class);
                return new DefaultJsonService(opendataService, this.httpService, serviceDescriptor);
            }
        }
        catch (IOException e) {
            throw new OpenDataException("Unable to find service " + opendataService, e);
        }
        throw new OpenDataException("Unable to find service " + opendataService);
    }

    @NonNull
    private Path getPath(String opendataService) {
        return Path.of(this.config.getDataPath(), opendataService + ".2.json");
    }

    private Optional<RoadworkData> loadCache(Path cachePath) {
        if (Files.exists(cachePath, new LinkOption[0])) {
            ObjectMapper objectMapper = new ObjectMapper();
            objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            try {
                RoadworkData roadworkData = objectMapper.readValue(cachePath.toFile(), RoadworkData.class);
                logger.info("Cache loaded");
                return Optional.of(roadworkData);
            }
            catch (IOException e) {
                logger.error("Unable to load cache {} {}", (Object)cachePath, (Object)e);
            }
        } else {
            return this.getMigratedData();
        }
        return Optional.empty();
    }

    private Optional<RoadworkData> getMigratedData() {
        logger.info("There is no cache, checking if there is old data");
        Map<String, RoadworkMigrationService> beansOfType = this.applicationContext.getBeansOfType(RoadworkMigrationService.class);
        Collection<RoadworkMigrationService> migrationServices = beansOfType.values();
        for (RoadworkMigrationService migrationService : migrationServices) {
            Optional<RoadworkData> roadworkData = migrationService.migrateData();
            if (!roadworkData.isPresent()) continue;
            logger.info("Migrating service {}", (Object)migrationService);
            migrationService.archive();
            this.save(roadworkData.get());
            return roadworkData;
        }
        return Optional.empty();
    }

    private static void applyFinishedStatus(RoadworkData roadworkData) {
        roadworkData.getRoadworks().values().stream().filter(Roadwork::isExpired).map(Roadwork::getSyncData).filter(Objects::nonNull).forEach(syncData -> syncData.setStatus(Status.Finished));
    }

    public void deleteCache() {
        String opendataService = this.config.getOpendataService();
        logger.info("deleteCache {}", (Object)opendataService);
        Path currentPath = this.getPath(opendataService);
        try {
            Files.deleteIfExists(currentPath);
        }
        catch (IOException e) {
            logger.error("Error deleting cache", e);
        }
    }

    public OpendataServiceManager(HttpService httpService, Config config, ApplicationContext applicationContext, SynchronizationService synchronizationService, ResourceService resourceService) {
        this.httpService = httpService;
        this.config = config;
        this.applicationContext = applicationContext;
        this.synchronizationService = synchronizationService;
        this.resourceService = resourceService;
    }
}

